<?php
// dashboard.php
session_start();

// 1. Security Check: Kick them to login if no active session
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

require 'db_connection.php';

$user_id = $_SESSION['user_id'];

// 2. Fetch basic user details
$stmt = $pdo->prepare("SELECT name, role_id FROM users WHERE user_id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

// 3. Fetch Active Shares for the Data Table
// This shows the manager exactly who has their documents and what the status is
$shares_stmt = $pdo->prepare("
    SELECT ds.recipient_name, ds.recipient_email, ds.status, ds.shared_at, d.title 
    FROM document_shares ds
    JOIN documents d ON ds.document_id = d.document_id
    WHERE ds.sender_id = ?
    ORDER BY ds.shared_at DESC LIMIT 10
");
$shares_stmt->execute([$user_id]);
$active_shares = $shares_stmt->fetchAll();

// Note: In a full system, you would have a query here to populate the "Select Document" 
// dropdown in the sharing modal based on what the user has uploaded.
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>NCDC | Executive Dashboard</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        /* Using the exact NCDC palette you provided */
        :root {
            --ncdc-green: #0A5C36; --ncdc-green-dark: #074729;
            --ncdc-yellow: #FCD116; --ncdc-red: #CE1126;
            --bg-app: #F4F7F9; --bg-surface: #FFFFFF;
            --text-primary: #1E293B; --text-secondary: #64748B;
            --border-light: #E2E8F0;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; font-family: 'Inter', sans-serif; font-size: 14px; }
        body { display: flex; background-color: var(--bg-app); color: var(--text-primary); height: 100vh; overflow: hidden; }

        /* Sidebar */
        .sidebar { width: 260px; background-color: var(--ncdc-green); color: #FFF; display: flex; flex-direction: column; }
        .sidebar-header { padding: 24px 20px; display: flex; align-items: center; gap: 15px; background: rgba(0,0,0,0.15); border-bottom: 1px solid rgba(255,255,255,0.1); }
        .sidebar-header i { font-size: 24px; color: var(--ncdc-yellow); }
        .sidebar-menu { list-style: none; padding: 20px 12px; }
        .sidebar-menu li { padding: 12px 14px; margin-bottom: 4px; border-radius: 6px; color: rgba(255,255,255,0.85); cursor: pointer; display: flex; align-items: center; gap: 12px; }
        .sidebar-menu li.active { background-color: rgba(255,255,255,0.15); color: var(--ncdc-yellow); font-weight: 600; border-left: 4px solid var(--ncdc-yellow); }
        .sidebar-menu li:hover:not(.active) { background-color: var(--ncdc-green-dark); }

        /* Workspace */
        .workspace { flex-grow: 1; display: flex; flex-direction: column; overflow-y: auto; }
        .topbar { background: var(--bg-surface); padding: 16px 28px; display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid var(--border-light); }
        
        .content { padding: 30px; max-width: 1200px; margin: 0 auto; width: 100%; }
        .header-actions { display: flex; justify-content: space-between; align-items: flex-end; margin-bottom: 24px; }
        .header-actions h1 { font-size: 24px; color: var(--text-primary); margin-bottom: 5px; }
        
        .btn { padding: 10px 16px; border-radius: 6px; font-weight: 600; cursor: pointer; border: none; display: inline-flex; align-items: center; gap: 8px; transition: 0.2s; }
        .btn-primary { background: var(--ncdc-green); color: white; }
        .btn-primary:hover { background: var(--ncdc-green-dark); }
        .btn-danger { background: var(--ncdc-red); color: white; }

        /* Tables & Panels */
        .panel { background: var(--bg-surface); border: 1px solid var(--border-light); border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.02); }
        .panel-header { padding: 16px 24px; border-bottom: 1px solid var(--border-light); background: #FAFAFA; display: flex; justify-content: space-between; align-items: center; }
        .data-table { width: 100%; border-collapse: collapse; text-align: left; }
        .data-table th { padding: 14px 24px; font-size: 12px; color: var(--text-secondary); border-bottom: 1px solid var(--border-light); }
        .data-table td { padding: 14px 24px; border-bottom: 1px solid var(--border-light); font-size: 13px; }
        
        .status { padding: 4px 10px; border-radius: 12px; font-size: 11px; font-weight: 600; text-transform: uppercase; }
        .status.shared { background: #FFFBEB; color: #D97706; }
        .status.downloaded { background: #ECFDF5; color: #059669; }
        .status.returned { background: #F1F5F9; color: #475569; }

        /* Modal Styles */
        .modal-overlay { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 100; align-items: center; justify-content: center; }
        .modal { background: white; padding: 30px; border-radius: 8px; width: 400px; max-width: 90%; }
        .modal h2 { margin-bottom: 15px; font-size: 18px; }
        .input-group { margin-bottom: 15px; }
        .input-group label { display: block; font-size: 12px; margin-bottom: 5px; font-weight: 600; color: var(--text-secondary); }
        .input-group input, .input-group select { width: 100%; padding: 10px; border: 1px solid var(--border-light); border-radius: 6px; }
    </style>
</head>
<body>

    <aside class="sidebar">
        <div class="sidebar-header">
            <i class="fas fa-shield-alt"></i>
            <div>
                <div style="font-weight: 700; font-size: 16px;">NCDC Portal</div>
                <div style="font-size: 11px; color: var(--ncdc-yellow);">Secure Operations</div>
            </div>
        </div>
        <ul class="sidebar-menu">
            <li class="active"><i class="fas fa-chart-pie"></i> Document Tracking</li>
            <li><i class="fas fa-search"></i> Leak Investigation</li>
            <li><i class="fas fa-users"></i> Staff Management</li>
            <li style="margin-top: auto; color: #FCA5A5;" onclick="window.location.href='logout.php'"><i class="fas fa-sign-out-alt"></i> Secure Logout</li>
        </ul>
    </aside>

    <main class="workspace">
        <header class="topbar">
            <div></div> <div style="font-weight: 600;"><i class="fas fa-user-circle"></i> <?php echo htmlspecialchars($user['name']); ?></div>
        </header>

        <div class="content">
            <div class="header-actions">
                <div>
                    <h1>Document Chain of Custody</h1>
                    <p style="color: var(--text-secondary);">Monitor active shares and external panel member access.</p>
                </div>
                <button class="btn btn-primary" onclick="openShareModal()"><i class="fas fa-share-alt"></i> Share Document</button>
            </div>

            <div id="statusMessage" style="display: none; padding: 15px; background: #ECFDF5; color: #059669; border-radius: 6px; margin-bottom: 20px; border: 1px solid #A7F3D0;"></div>

            <div class="panel">
                <div class="panel-header">
                    <h3 style="font-size: 15px;"><i class="fas fa-list"></i> Active External Links</h3>
                </div>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Document Name</th>
                            <th>Recipient</th>
                            <th>Email</th>
                            <th>Date Shared</th>
                            <th>Live Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($active_shares)): ?>
                            <tr><td colspan="5" style="text-align: center; color: var(--text-secondary);">No active document shares found.</td></tr>
                        <?php else: ?>
                            <?php foreach ($active_shares as $share): ?>
                                <tr>
                                    <td style="font-weight: 600;"><?php echo htmlspecialchars($share['title']); ?></td>
                                    <td><?php echo htmlspecialchars($share['recipient_name']); ?></td>
                                    <td><?php echo htmlspecialchars($share['recipient_email']); ?></td>
                                    <td><?php echo date('M d, Y H:i', strtotime($share['shared_at'])); ?></td>
                                    <td><span class="status <?php echo strtolower($share['status']); ?>"><?php echo htmlspecialchars($share['status']); ?></span></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <div class="modal-overlay" id="shareModal">
        <div class="modal">
            <h2>Generate Secure Link</h2>
            <form id="shareForm">
                <div class="input-group">
                    <label>Select Document to Share</label>
                    <select id="document_id" required>
                        <option value="">-- Choose a document --</option>
                        <option value="1">S.4 Mathematics Curriculum (Draft)</option>
                        <option value="2">O-Level Science Lab Guidelines</option>
                    </select>
                </div>
                <div class="input-group">
                    <label>Recipient Full Name</label>
                    <input type="text" id="recipient_name" placeholder="e.g., John Doe" required>
                </div>
                <div class="input-group">
                    <label>Recipient Official Email</label>
                    <input type="email" id="recipient_email" placeholder="john@example.com" required>
                </div>
                <div style="display: flex; gap: 10px; margin-top: 20px;">
                    <button type="button" class="btn" style="background: #E2E8F0; width: 50%;" onclick="closeShareModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary" style="width: 50%; justify-content: center;" id="btnShare">Send Magic Link</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        const modal = document.getElementById('shareModal');
        const form = document.getElementById('shareForm');
        const statusMsg = document.getElementById('statusMessage');

        function openShareModal() { modal.style.display = 'flex'; }
        function closeShareModal() { modal.style.display = 'none'; form.reset(); }

        form.addEventListener('submit', function(e) {
            e.preventDefault();
            const btn = document.getElementById('btnShare');
            btn.textContent = 'Generating...';
            btn.disabled = true;

            const formData = new FormData();
            formData.append('document_id', document.getElementById('document_id').value);
            formData.append('recipient_name', document.getElementById('recipient_name').value);
            formData.append('recipient_email', document.getElementById('recipient_email').value);

            // Calls the share_document.php script we wrote earlier
            fetch('share_document.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                closeShareModal();
                btn.textContent = 'Send Magic Link';
                btn.disabled = false;
                
                statusMsg.style.display = 'block';
                if (data.status === 'success') {
                    statusMsg.style.color = '#059669';
                    statusMsg.style.backgroundColor = '#ECFDF5';
                    statusMsg.style.borderColor = '#A7F3D0';
                    statusMsg.textContent = 'Success: Magic Link has been generated and emailed to the recipient.';
                    // Refresh the page after 2 seconds to show the new table row
                    setTimeout(() => window.location.reload(), 2000);
                } else {
                    statusMsg.style.color = '#DC2626';
                    statusMsg.style.backgroundColor = '#FEF2F2';
                    statusMsg.style.borderColor = '#FCA5A5';
                    statusMsg.textContent = 'Error: ' + data.message;
                }
            })
            .catch(error => {
                btn.textContent = 'Send Magic Link';
                btn.disabled = false;
                alert('Connection error. Could not contact server.');
            });
        });
    </script>
</body>
</html>